import groovy.json.JsonBuilder
import groovy.json.JsonSlurper

import java.net.http.HttpClient
import java.net.http.HttpRequest
import java.net.http.HttpResponse
import java.nio.ByteBuffer
import java.nio.charset.StandardCharsets
import java.util.concurrent.CompletableFuture
import java.util.concurrent.CompletionStage
import java.util.concurrent.Flow.Subscription
import java.util.function.Consumer

class OpenAISSEBodySubscriber implements HttpResponse.BodySubscriber<Void> {
    private final Consumer contentConsumer
    private final JsonSlurper jsonSlurper
    private final CompletableFuture<Void> future
    private volatile Subscription subscription
    private volatile String deferredText

    OpenAISSEBodySubscriber(Consumer<String> contentConsumer, JsonSlurper jsonSlurper) {
        this.contentConsumer = contentConsumer
        this.jsonSlurper = jsonSlurper
        this.future = new CompletableFuture<>()
        this.deferredText = null
    }

    @Override
    void onNext(List<ByteBuffer> buffers) {
        try {
            String deferredText = this.deferredText
            for (ByteBuffer buffer : buffers) {
                String s = deferredText + StandardCharsets.UTF_8.decode(buffer)
                def tokens = s.split('\n\n', -1)
                for (int i = 0; i < tokens.length - 1; i++) {
                    String message = tokens[i]
                    def lines = message.split('\n')

                    StringBuilder stringBuilder = new StringBuilder()
                    for (String line : lines) {
                        if (line.startsWith('data:')) {
                            stringBuilder.append(line.substring(5))
                        }
                    }
                    contentConsumer.accept(stringBuilder.toString())
                }

                deferredText = tokens[tokens.length - 1]
            }

            this.deferredText = deferredText
            subscription.request(1)
        } catch (Exception ex) {
            future.completeExceptionally(ex)
            subscription.cancel()
        }
    }

    @Override
    void onSubscribe(Subscription subscription) {
        this.subscription = subscription
        try {
            this.deferredText = ''
            this.subscription.request(1)
        } catch (Exception ex) {
            future.completeExceptionally(ex)
            this.subscription.cancel()
        }
    }

    @Override
    void onError(Throwable throwable) {
        future.completeExceptionally(throwable)
    }

    @Override
    void onComplete() {
        try {
            future.complete(null)
        } catch (Exception ex) {
            future.completeExceptionally(ex)
        }
    }

    @Override
    CompletionStage<Void> getBody() {
        return this.future
    }
}

@groovy.transform.CompileStatic 
class OpenAiConnectionHandler extends Secrets {
    static chat(Map parameters, HttpClient httpClient, JsonSlurper jsonSlurper, Consumer contentConsumer) {
        def apiVersion = getSecret('openai_api_version')
        def openaiResourceName = getSecret('openai_resource_name')
        def apiKey = getSecret('openai_api_key')
        def deploymentId = getSecret('openai_deployment_id')

        def body = [
            messages   : parameters.messages,
            temperature: parameters.temperature ?: parameters.temperature,
            max_tokens : parameters.max_tokens ?: 16384,
            stream  : true
        ]

        def uri = URI.create("https://${openaiResourceName}.openai.azure.com/openai/deployments/${deploymentId}/chat/completions?api-version=${apiVersion}")
        def request = HttpRequest.newBuilder(uri)
                .header('api-key', apiKey)
                .header('Content-Type', 'application/json')
                .POST(HttpRequest.BodyPublishers.ofString(new JsonBuilder(body).toString()))
                .build()
        def future = httpClient.sendAsync(request, { responseInfo ->
            if (responseInfo.statusCode() == 200) {
                return new OpenAISSEBodySubscriber(contentConsumer, jsonSlurper)
            } else {
                throw new RuntimeException('Request failed ' + responseInfo.statusCode())
            }
        })
    }
}
